<?php
// CLASS MANAGING E-MAIL CAMPAIGN SYSTEMS

class pcma_email_campaigns {
    
    public static $esipp_transient_name = 'dike_ec_%PERIOD%_emails_sent'; // (string) name for emails-sent-in-past-period transient with placeholder to be replaced with get_option('pcma_bulk_mail_limit_type', 'hourly')
    
    
    
    // register custom post type where campaigns will be stored
    public static function register_cpt_n_ct() {
        if(!pcma_is_active()) {
            return false;    
        }
        
        // which is the minimum role to manage it?
        $cuc_edit = pc_wpuc_static::current_wp_user_can_edit_pc_user('some');
        if(!$cuc_edit) {
            $capability = get_option('pg_min_role_tmu', 'edit_pages');
        }
        else {
            // get first user role to be used in add_menu_page()
            $curr_user = wp_get_current_user();
            $capability = (count((array)$curr_user->roles)) ? reset($curr_user->roles) : get_option('pg_min_role_tmu', 'edit_pages');
        }
        
        $labels = array( 
            'name'              => __('Campaigns', PCMA_ML),
            'singular_name'     => __('Campaign', PCMA_ML),
            'add_new'           => __('Add New Campaign', PCMA_ML),
            'add_new_item'      => __('Add New Campaign', PCMA_ML),
            'edit_item'         => __('Edit Campaign', PCMA_ML),
            'new_item'          => __('New Campaign', PCMA_ML),
            'view_item'         => __('View Campaign', PCMA_ML),
            'search_items'       => __('Search Campaigns', PCMA_ML),
            'not_found'         => __('No Campaigns found', PCMA_ML),
            'not_found_in_trash'=> __('No Campaigns found in Trash', PCMA_ML),
            'menu_name'         => __('E-mail Campaigns', PCMA_ML),
        );
        
        $svg_logo = 'data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHZpZXdCb3g9IjAgMCAxNiAxNiI+PHBvbHlnb24gcG9pbnRzPSIxMy4wMiAxIDMuMTMgMSAxIDIuNyAxLjE2IDIuNyAxNSAyLjcgMTMuMDIgMSIgc3R5bGU9ImZpbGw6IzljYTJhNyIvPjxwYXRoIGQ9Ik0xLDMuMzJWMTVIMTVWMy4zMlpNMTIsNi4xLDguMjEsOS40M2gwQS4yNC4yNCwwLDAsMSw4LDkuNTVhLjU3LjU3LDAsMCwxLS4yMy0uMTJMNCw2LjFabS04Ljg0LjgyTDUuNTEsOSwzLjE5LDExLjMxWm0uNjUsNS4yNSwyLjUtMi41LjYuNTNhMS4zMywxLjMzLDAsMCwwLDEsLjQyLDEuMzEsMS4zMSwwLDAsMCwxLS40MWwuNjItLjU0TDEyLDEyLjE3Wk0xMC40NSw5bDIuMzYtMnY0LjMyWiIgc3R5bGU9ImZpbGw6IzljYTJhNyIvPjwvc3ZnPg==';

        $args = array( 
            'labels'            => $labels,
            'hierarchical'      => false,
            'public'            => false,
            'show_ui'           => true,
            'show_in_menu'      => true,
            'menu_icon'         => $svg_logo,
            'menu_position'     => 48,
            'show_in_nav_menus' => false,
            'publicly_queryable'=> false,
            'exclude_from_search' => true,
            'has_archive'       => false,
            'query_var'         => false,
            'can_export'        => true,
            'rewrite'           => false,
            'supports'          => array('title', 'editor'),
            'taxonomies'        => array('pcma_email_cmp_cats'), 
            'capabilities'      => array(
                'edit_post'          => $capability,
                'read_post'          => $capability,
                'delete_post'        => $capability,
                'edit_posts'         => $capability,
                'edit_others_posts'  => $capability,
                'delete_posts'       => $capability,
                'publish_posts'      => $capability,
                'read_private_posts' => $capability
            ),
        ); 
        register_post_type('pcma_email_campaigns', $args);
        

        //////
	
        
        $labels = array( 
            'name'              => __('Campaign Categories', PCMA_ML),
            'singular_name'     => __('Campaign Category', PCMA_ML),
            'search_items'      => __('Search Campaign Categories', PCMA_ML),
            'popular_items'     => NULL,
            'all_items'         => __('All Campaign Categories', PCMA_ML),
            'parent_item'       => __('Parent Campaign Category', PCMA_ML),
            'parent_item_colon' => __('Parent Campaign Category:', PCMA_ML),
            'edit_item'         => __('Edit Campaign Category', PCMA_ML),
            'update_item'       => __('Update Campaign Category', PCMA_ML),
            'add_new_item'      => __('Add New Campaign Category', PCMA_ML),
            'new_item_name'     => __('New Campaign Category', PCMA_ML),
            'separate_items_with_commas'    => __('Separate campaign categories with commas', PCMA_ML),
            'add_or_remove_items'           => __('Add or remove campaign categories', PCMA_ML),
            'choose_from_most_used'         => __('Choose from most used campaign categories', PCMA_ML),
            'menu_name'         => __('Campaign Categories', PCMA_ML),
        );

        $args = array( 
            'labels' 			=> $labels,
            'public' 			=> false,
            'show_in_nav_menus' => false,
            'show_ui' 			=> true,
            'show_tagcloud'		=> false,
            'hierarchical' 		=> true,
            'rewrite' 			=> false,
            'query_var' 		=> true,
            'update_count_callback' => '_update_generic_term_count'
        );
        register_taxonomy('pcma_email_cmp_cats', array('pcma_email_campaigns'), $args);
        
    }
    
    
    // customize CPT messages
    public static function cpt_updated_messages( $messages ) {
        global $post;

        $messages['pcma_email_campaigns'] = array(
            0 => '', // Unused. Messages start at index 1.
            1 => __('Campaign updated', PCMA_ML),
            2 => __('Campaign updated', PCMA_ML),
            3 => __('Campaign deleted', PCMA_ML),
            4 => __('Campaign updated', PCMA_ML),
            /* translators: %s: date and time of the revision */
            5 => isset($_GET['revision']) ? sprintf( __('Campaign restored to revision from %s', PCMA_ML), wp_post_revision_title( (int) $_GET['revision'], false ) ) : false,
            6 => __('Campaign scheduled', PCMA_ML),
            7 => __('Campaign saved', PCMA_ML),
            8 => __('Campaign submitted', PCMA_ML),
            9 => sprintf( __('Campaign scheduled for: <strong>%1$s</strong>', PCMA_ML), date_i18n( __('M j, Y @ G:i' ), strtotime( $post->post_date ))),
            10 => __('Campaign draft updated', PCMA_ML),
        );

        return $messages;
    }
    
    
    
    
    
    /*
     * Given an array of keyword => meta_name associations, returns the related values array already managed 
     */
    public static function get_campaign_metas($campaign_id, $assoc) {
        $to_return = array();
        
        foreach($assoc as $arr_key => $meta_key) {
            $val = get_post_meta($campaign_id, $meta_key, true);

            // manage attachments value to be an array
            if($meta_key == 'pcma_ecb_attach' && !empty($val)) {
                $val = explode(',', $val);
            }

            // already arrange extra-mails as an array
            elseif($meta_key == 'pcma_ecb_extra_emails') {
                $val = self::extra_emails_val_to_array($val);
            }

            // processed and failed must be decompressed and set to array
            if(in_array($meta_key, array('pcma_ecb_receivers_processed', 'pcma_ecb_receivers_failed'))) {
                if(!empty($val) && !is_array($val)) {
                    $val = pc_static::decompress_data($val);
                }
                $val = (empty($val)) ? array() : (array)$val;
            }

            
            // PCMA-FILTER - allows extra control over pcma_email_campaigns::get_campaign_metas() value retrieval and management
            $val = apply_filters('pcma_get_campaign_metas_val', $val, $meta_key, $campaign_id);
            
            $to_return[$arr_key] = $val;     
        }
        
        return $to_return;
    }
    
    
    
    
    /*
     * knows whether a campaign can still be edited (if is not complete or in progress)
     * @param (false|string) $post_status - if post status is already available, why get it again?
     */
    public static function campaign_can_be_edited($campaign_id, $post_status = false) {
        if(!$post_status) {
            $post_status = get_post_status($post_status);    
        }
        
        // not published or scheduled
        if(!in_array($post_status, array('publish', 'future'))) {
            return true;    
        }
        
        return (get_post_meta($campaign_id, 'pcma_ecb_start_time', true)) ? false : true;
    }
    

    
    
    /*
     * Knows whether system can send e-mails in this moment
     * @return (bool) false if the engine already reached the hourly threshold
     */
    public static function can_send_emails() {
        $emails_limit_val = (int)get_option('pcma_bulk_mail_limit_num', 400);
        
        $transient_name = str_replace('%PERIOD%', get_option('pcma_bulk_mail_limit_type', 'hourly'), self::$esipp_transient_name);
        $emails_sent_in_target_period = (int)get_transient($transient_name);
        
        return ($emails_sent_in_target_period < $emails_limit_val) ? true : false;
    }
    
    
    
    
    // get attachment code block for admin builder
    public static function admin_builder_attach_code($file_id, $campaign_id) {
        $file_id = (int)$file_id;
        $thumb_data = wp_get_attachment_image_src($file_id, array(40, 40), true);
        $wp_file_icon_class = (strpos($thumb_data[0], 'wp-includes/images/media') !== false) ? 'pcma_ecb_wp_file_icon' : '';
        
        $file_url = wp_get_attachment_url($file_id);
        $url_parts_arr = explode('/', $file_url);

        $file_metas = wp_get_attachment_metadata($file_id);
        if(!$file_metas) {
            $bytes = wp_filesize(get_attached_file($file_id));
        } else {
            $bytes = $file_metas['filesize'];   
        }
        
        // human filesize
        $decimals = 2;
        
        $sizes = array('Bytes','KB','MB','GB','TB','PB','EB','ZB','YB');
        $factor = floor((strlen($bytes) - 1) / 3);
        if($factor < 0) {
            $factor = 0;    
        }
        
        $val = sprintf("%.{$decimals}f", $bytes / pow(1024, $factor));

        // remove precise values
        if($decimals) {
            if(!(int)substr($val, ($decimals * -1))) {
                $arr = explode('.', $val);
                $val = $arr[0];
            }
        }
        $filesize = $val .' '. $sizes[$factor];
        
        $del_btn_part = (self::campaign_can_be_edited($campaign_id)) ? 'dashicons-no-alt" title="'. esc_attr__('remove attachment', PCMA_ML) .'"' : '"';
        
        return '
        <li data-file-id="'. $file_id .'">
            <figure class="'. $wp_file_icon_class .'" style="background-image: url('. esc_attr($thumb_data[0]) .');"></figure>
            <i class="dashicons '. $del_btn_part .'></i>
            <a class="dashicons dashicons-external" href="'. $file_url .'" target="_blank" title="'. esc_attr__('open in a new tab', PCMA_ML) .'"></a>
            <span>'. end($url_parts_arr) .' <em>('. $filesize .') (#'. $file_id .')</em></span>
        </li>';   
    }
    
    
    
    
    /*
     * Returns HTML code for builder's user filter
     * @param (false|array) $all_fields > fields array got from pc_form class
     */
    public static function admin_builder_user_filter_code($filter_data = false, $all_fields = false) {
        if($filter_data && !isset($all_fields[ (string)$filter_data['field'] ])) {
            return false;    
        }
        
        if(!is_array($filter_data)) {
            $filter_data = array(
                'label' => '%%LABEL%%',
                'field' => '%%FIELD%%',
                'cond'  => '',
                'val'   => '',
            );
        }
        else {
            $filter_data['label'] = $all_fields[ $filter_data['field'] ]['label'];   
        }
    
        return '
        <li>
            <span class="pc_del_field dashicons dashicons-no-alt" title="'. __('remove condition', PCMA_ML) .'"></span>
            <span>
                <input type="hidden" name="pcma_ecb_auf_f[]" value="'. $filter_data['field'] .'" autocomplete="off" />
                '. $filter_data['label'] .'
            </span>
            <select name="pcma_ecb_auf_cond[]" autocomplete="off">
                <option value="equal" '. selected('equal', $filter_data['cond'], false) .'>'. __('is equal to', PC_ML) .'</option>
                <option value="different" '. selected('different', $filter_data['cond'], false) .'>'. __('is different from', PC_ML) .'</option>	
                <option value="bigger" '. selected('bigger', $filter_data['cond'], false) .'>'. __('is greater than', PC_ML) .'</option>	
                <option value="smaller" '. selected('smaller', $filter_data['cond'], false) .'>'. __('is lower than', PC_ML) .'</option>
                <option value="like" '. selected('like', $filter_data['cond'], false) .'>'. __('contains', PC_ML) .'</option>
                <option value="not_like" '. selected('not_like', $filter_data['cond'], false) .'>'. __('does not contain', PC_ML) .'</option>
            </select>
            <input type="text" name="pcma_ecb_auf_val[]" value="'. esc_attr($filter_data['val']) .'" autocomplete="off" />
        </li>';
    }
    
    
    
    
    /* 
     * Get involved pvtContent users 
     *
     * @param (int|false) $campaign_id - the campaign ID or false if data is manually passed in the second argument
     * @param (false|array) $receivers_data - false if data has to be retrieved via campaign id, otherwise the data array
     *
     * @return (array) multidimensional array having user ID as key and array(userame, email) as val
     */
    public static function get_involved_users($campaign_id = false, $receivers_data = false) {
        global $pc_users;
        $data = array();
        
        // retrieve data
        if($campaign_id) {
            $meta_keys = array(
                'pcma_ecb_include_pc_cats',
                'pcma_ecb_exclude_pc_cats',
                'pcma_ecb_incl_disabled_users',
                'pcma_ecb_user_filters',
                'pcma_ecb_auf_operator',
                'pcma_ecb_extra_user_ids',
            );      
            
            foreach($meta_keys as $mk) {
                $data[$mk] = get_post_meta($campaign_id, $mk, true);    
            }
        }
        else {
            $data = $receivers_data;
            if(empty($data)) {
                return array();        
            }
        }
        
        
        // no specified cat? stop
        if(empty($data['pcma_ecb_include_pc_cats'])) {
            return array();    
        }
        
        
        // essential query parameters
        $search = array(
            'base' => array(
                'relation' => 'AND',
                array('key'=>'email', 'operator'=>'!=', 'val'=>''),
            )
        );
        
        // any user ID specifically passed and to exclude?
        if(!empty($data['pcma_ecb_extra_user_ids'])) {
            $search['base'][] = array('key'=>'id', 'operator'=>'NOT IN', 'val'=>(array)$data['pcma_ecb_extra_user_ids']);  
        }
        
        // exclude any category?
        if(!empty($data['pcma_ecb_exclude_pc_cats'])) {
            $search['base'][] = array('key'=>'categories', 'operator'=>'NOT IN', 'val'=>(array)$data['pcma_ecb_exclude_pc_cats']);        
        }
        
        
        
        // filters?
        if(!empty($data['pcma_ecb_user_filters'])) {
            $search_block = array(
                'relation' => (count((array)$data['pcma_ecb_user_filters']) < 2) ? 'AND' : $data['pcma_ecb_auf_operator'],
            );
            
            foreach($data['pcma_ecb_user_filters'] as $filter) {	

                // operator translation
                switch($filter['cond']) {
                    case 'different': $op = '!='; break;
                    case 'bigger'	: $op = '>'; break;
                    case 'smaller'	: $op = '<'; break;
                    case 'like'		: $op = 'LIKE'; break;
                    case 'not_like'	: $op = 'NOT LIKE'; break;
                    case '='        : 
                    default         : $op = '='; break;	
                }
 
                $search_block[] = array(
                    'key'       => $filter['field'], 
                    'val'       => (in_array($op, array('LIKE', 'NOT LIKE'))) ? "%". stripslashes($filter['val']) ."%" : stripslashes($filter['val']), 
                    'operator'  => $op
                );
            }
            
            $search[] = $search_block;
        }
        
        $args = array(
            'to_get'        => array('id', 'username', 'email'),
            'limit' 	    => -1,
            'status'        => ($data['pcma_ecb_incl_disabled_users']) ? false : '1',
            'categories'    => (empty($data['pcma_ecb_include_pc_cats']) || in_array('all', (array)$data['pcma_ecb_include_pc_cats'])) ? false : $data['pcma_ecb_include_pc_cats'],
            'search'	    => $search,
            'orderby'       => 'username',
        );
        
        
        // PCMA-FILTER - allow parameters customization for mail campaign users query
        //// $campaign_id is filled if data must be retrieved via metas, $receivers_data, if filled, contains ajax-fetched fields val
        $args = apply_filters('pcma_email_campaign_receivers_query', $args, $campaign_id, $receivers_data);
        
        // consider add_filter('pcma_qm_cat_query'
        return $pc_users->get_users($args);
    }
    
    
    

    /*
     * Given the extra emails textarea value, returns the e-mails array
     */
    public static function extra_emails_val_to_array($val) {
        if(!empty($val) && !filter_var($val, FILTER_VALIDATE_EMAIL) && strpos($val, "\n") === false) {
            $val = pc_static::decompress_data($val);
        }
        return (empty($val)) ? array() : (array)preg_split('/\r\n|[\r\n]/', (string)$val);
    }
    
    
    

    /*
     * Passing receivers + extra user ids + extra emails, return the overall array (being sure no duplicates are among receivers and user IDs)
     */
    public static function overall_receivers($receivers, $extra_uid, $extra_emails) {
        if(!is_array($receivers)) {
            $receivers = array();    
        }
        
        if(is_array($extra_uid) && !empty($extra_uid)) {
            $receivers = array_unique(array_merge($receivers, $extra_uid));     
        }
        
        if(is_array($extra_emails) && !empty($extra_emails)) {
            $receivers = array_merge($receivers, $extra_emails);     
        }
        
        return $receivers;
    }
    
    
    
    
    /* 
     * Returns the mails-sent-in-past-hour transient expiration in H:i format
     */
    public static function get_eph_transient_expiration() {
        $expiration_gmt_time = get_option('_transient_timeout_'. self::$esiph_transient_name);
        if(!$expiration_gmt_time) {
            return false;    
        }
        
        return get_date_from_gmt(date('Y-m-d H:i:s', $expiration_gmt_time), 'H:i'); 
    }
    
    
    

    /*
     * Passing receivers, processed and failed sendings array, output the details table
     */
    public static function progress_details_table($receivers, $processed, $failed) {
        global $pc_users;
        if(empty($receivers)) {
            return '';   
        }

        
        // get users data
        $args = array(
            'user_id'   => $receivers,
            'limit'     => count($receivers),
            'to_get'    => array('id', 'email', 'username'),
        );
        $user_data = $pc_users->uid_as_user_data_array_key($pc_users->get_users($args));
        
        $code = '
        <div class="pcma_ecb_progress_details pc_displaynone">
            <table class="widefat striped pcma_ecb_show_receivers_list_table">
                <thead>
                    <tr>
                        <th><strong>#</strong></th>
                        <th><strong>'. __('Username', PC_ML) .'</strong></th>
                        <th><strong>'. __('E-mail', PC_ML) .'</strong></th>
                        <th><strong>'. __('Processed', PC_ML) .'</strong> <span class="pc_del_field dashicons dashicons-no-alt" onclick="window.lcwpm_close()"></span></th>
                    </tr>
                </thead>
                <tbody>';
        
                foreach($receivers as $index => $receiver) {
                    $id         = (is_numeric($receiver)) ? $receiver : '';
                    $username   = (isset($user_data[ $receiver ])) ? $user_data[ $receiver ]['username'] .' <small>ID '. $id .'</small>' : '';
                    $email      = ($username) ? $user_data[ $receiver ]['email'] : $receiver;
                    
                    $status_icon = '';
                    $fail_reason = '';
                    
                    $process_date   = (is_array($processed) && isset($processed[ $receiver ])) ? $processed[ $receiver ] : '';
                    
                    if($process_date) {
                        $process_date   = '<span class="dashicons dashicons-clock"></span> '. get_date_from_gmt(date('Y-m-d H:i:s', $process_date), 'Y/m/d H:i') .'<br/>';
                        
                        $fail_reason    = (is_array($failed) && isset($failed[ $receiver ])) ? $failed[ $receiver ] : '';
                        $success_reason = __('Sent successfully!', PCMA_ML);
                        
                        $status_icon    = ($fail_reason) ? 
                            '<span class="dashicons dashicons-dismiss"></span> '. $fail_reason : 
                            '<span class="dashicons dashicons-yes-alt"></span> '. $success_reason;
                    }
                    
                    $code .= '
                    <tr>
                        <td>'. ($index + 1) .'</td>
                        <td>'. $username .'</td>
                        <td>'. $email .'</td>
                        <td>
                            '. $process_date. '
                            '. $status_icon .'
                        </td>
                    </tr>';
                }
                
        return $code .'
                </tbody>
            </table>
        </div>';
    }
    
    
    
    
    /* 
     * Outputs the javascript snippet to manage campaign clone
     */
    public static function campaign_clone_js_code() {
        ?>
        $(document).on('click', '.pcma_ecb_clone_cmp:not(.pcma_opacity_loader)', function() {
            const $btn = $(this),
                  curr_title = (!$('.subsubsub').length) ? $('#title').val() : $btn.parents('td').first().find('.row-title').text();

            let new_title = prompt(`<?php esc_attr_e('Which internal name to use?', PCMA_ML) ?>`, curr_title.trim() + ' [clone]');    

            if(!new_title) {
                return false;
            }
            if(!new_title.trim()) {
                lc_wp_popup_message('error', "<?php esc_attr_e('Please use a valid title', PCMA_ML) ?>");  
                return false;
            }


            $btn.addClass('pcma_opacity_loader');

            let data = {
                action      : 'pcma_clone_campaign',
                campaign_id : parseInt($btn.data('cmp-id'), 10),
                new_title   : new_title,
                nonce       : '<?php echo wp_create_nonce('lcwp_nonce') ?>'
            };
            $.post(ajaxurl, data, function(response) {
                try {
                    const resp = JSON.parse(response);

                    if(resp.status != 'success') {
                        lc_wp_popup_message('error', resp.mess);	
                        return false;
                    }

                    lc_wp_popup_message('success', "<?php esc_attr_e('Campaign successfully cloned!', PCMA_ML) ?>");
                    setTimeout(function() {
                        window.location.href = resp.new_cmp_editor_url;
                    }, 2000);
                }
                catch(e) {
                    console.error(e);
                    lc_wp_popup_message('error', "<?php esc_attr_e('Error cloning the campaign', PCMA_ML) ?>");
                }
            })
            .fail(function(e) {
                if(e.status) {
                    console.error(e);
                    lc_wp_popup_message('error', "<?php esc_attr_e('Error cloning the campaign', PCMA_ML) ?>");
                }    
            })
            .always(function() {
                $btn.removeClass('pcma_opacity_loader');
            });
        });
        <?php
    }
    
    

    
    /* 
     * Outputs the javascript snippet to manage sent-email-counter reset
     */
    public static function sent_emails_counter_reset_js_code() {
        ?>
        $(document).on("click", "#pcma_ec_reset_set:not(.pcma_ec_reset_set_is_acting)", function() {
            if(!confirm(`<?php esc_attr_e('Do you really want to reset the sent e-mails counter?', PCMA_ML) ?>`)) {
                return false;    
            }

            const $btn = $(this);
            $btn.addClass(`pcma_ec_reset_set_is_acting`).fadeTo(200, 0.5);

            let data = {
                action   : `pcma_reset_sent_email_counter`,
                nonce    : `<?php echo wp_create_nonce('lcwp_nonce') ?>`,
            };

            $.post(ajaxurl, data, function(response) {
                const resp = $.trim(response);    

                if(resp == `success`) {
                    lc_wp_popup_message(`success`, `<?php esc_attr_e('Counter successfully resetted!', PCMA_ML) ?>`);

                    setTimeout(function() {
                        window.location.reload();
                    }, 1000);
                } 
                else {
                    lc_wp_popup_message(`error`, resp); 
                }
            })
            .fail(function(e) {
                if(e.status) {
                    console.error(e);
                    lc_wp_popup_message(`error`, `error processing the operation`);
                }
            })
            .always(function() {
                $btn.removeClass(`pcma_ec_reset_set_is_acting`).fadeTo(200, 1);
            });
        });
        <?php
    }
    
    
    
    
    /* 
     * Returns the mails-sent-in-past-hour transient expiration in H:i format
     */
    public static function get_sending_transient_expiration() {
        $emails_limit_type = get_option('pcma_bulk_mail_limit_type', 'hourly');
        $transient_name = str_replace('%PERIOD%', get_option('pcma_bulk_mail_limit_type', 'hourly'), self::$esipp_transient_name);
        
        $expiration_gmt_time = get_option('_transient_timeout_'. $transient_name);
        if(!$expiration_gmt_time) {
            return false;    
        }
        
        $format = ($emails_limit_type == 'hourly') ? 'H:i' : 'd/m H:i'; 
        $conjunction = ($emails_limit_type == 'hourly') ? __('at', PCMA_ML) : __('the', PCMA_ML); 
        $date = get_date_from_gmt(date('Y-m-d H:i:s', $expiration_gmt_time), $format);
        
        return ' '. $conjunction .' '. str_replace(' ', ' '. __('at', PCMA_ML) .' ', $date); 
    }
    
    
    
    
    /* 
     * Eventually returns the warning related e-mail threshold progress or limit 
     * @return (string)
     */
    public static function get_esipp_warning() {
        $text = '';
        
        if(!self::can_send_emails()) {
            $text = '
            '. __('Your sending limit has been reached, next e-mails will be processed', PCMA_ML) .' '. self::get_sending_transient_expiration();        
        }
        else {
            $emails_limit_type = get_option('pcma_bulk_mail_limit_type', 'hourly');

            $transient_name = str_replace('%PERIOD%', get_option('pcma_bulk_mail_limit_type', 'hourly'), self::$esipp_transient_name);
            $emails_sent_in_target_period = (int)get_transient($transient_name);   

            if($emails_sent_in_target_period) {
                $emails_limit_val = (int)get_option('pcma_bulk_mail_limit_num', 400);
                $diff = $emails_limit_val - $emails_sent_in_target_period;

                $text = $emails_sent_in_target_period .' '. __('e-mails already sent. You still have', PCMA_ML) .' '. $diff .' '. __('available until', PCMA_ML) .' '. self::get_sending_transient_expiration();        
            }
        }


        if($text) {
            $reset_txt = '<a id="pcma_ec_reset_set" class="page-title-action" href="javascript:void(0)">'. __('reset counter', PCMA_ML) .'</a>'; 
            return '<span>'. $text .' '. $reset_txt .'</span>';     
        }
        return '';     
    }
    
    
    
       
    /////////////////////////////////////////////////////////////////////
    
    
    
    
    // defines a WP cron schedule period to run the process every 5 minutes
    public static function set_schedule_timing($schedules) {
        if(!isset($schedules["pcma_ec_timing"])){
            
            $schedules["pcma_ec_timing"] = array(
                'interval'  => 60 * 5,
                'display'   => __('Every 5 minutes', PCMA_ML)
            );
        }
        return $schedules;
    }

    
    
    
    // register the cron job in the WP engine
    public static function register_cron_job() {
        if(!wp_next_scheduled('pcma_process_campaigns')) {
            
            // scheduling at precise times (5/10/15/etc minutes)
            $closest_5min = ceil((int)date_i18n('i') / 5) * 5;
            $hour = gmdate('G');
            
            if($closest_5min == 60) {
                $closest_5min = 0;
                $hour = $hour + 1;
            }
            
            $response = wp_schedule_event(strtotime(gmdate('Y-m-d ') . zeroise($hour, 2).':'. zeroise($closest_5min, 2) .':00'), 'pcma_ec_timing' , 'pcma_process_campaigns');
        }
    }
    
    

    
    // process campaigns - callback for wp_schedule_event()
    public static function wp_schedule_process_campaigns() {
        $time_limit = 60 * 9.5; 
        ini_set('max_execution_time', $time_limit);
        set_time_limit($time_limit);
        
        $transient_name = 'pcma_processing_email_campaigns';
        if(get_transient($transient_name)) {
            return 'PCMA campaigns process - process still running';    
        }
        set_transient($transient_name, 1, $time_limit);
        
        $class = new pcma_email_campaigns;
        $class->process_campaigns();
        
        delete_transient($transient_name);
    }
    
    

    
    // process campaigns - manual mode
    public static function manual_campaigns_process() {
        if(!isset($_GET['pcma_process_campaigns'])) {
            return false;   
        }
        
        self::wp_schedule_process_campaigns();
    }
        
        
        
    
    /////////////////////////////////////////////////////////////////////
    
    
    
    
    /////////////////////////////////////////////////////////////////////
    // NON-STATIC PART OF THE CLASS /////////////////////////////////////
    /////////////////////////////////////////////////////////////////////
    
    protected $emails_limit_val; // (int) +
    protected $emails_limit_type; // (string) how many emails user allowed to send in a target period
    
    protected $emails_sent_in_past_period = 0; // (int) how many e-mails author sent in its target period
    protected $emails_sent_now = 0; // (int) how any email have been sent in this session
    
    protected $attach_data = array(); // (array) static attachments data storage to avoid calls for each user. Has attachment ID as key 
    
    
    public function __construct() {
        $this->emails_limit_val = (int)get_option('pcma_bulk_mail_limit_num', 400);
        $this->emails_limit_type = get_option('pcma_bulk_mail_limit_type', 'hourly');
        
        $target_transient_name = str_replace('%PERIOD%', $this->emails_limit_type, self::$esipp_transient_name);
        $this->emails_sent_in_past_period = (int)get_transient($target_transient_name);
    }
    
    
        
    
    /* 
     * Fetches campaigns that needs to be managed right now
     *
     * @param (bool) $store_receivers - whether to statically store fetched receivers into the post meta (eg. on campaign process start) 
     * @return (array) associative array having the campaign ID as key and its data as values, already sorted by schedule date
     */
    protected function get_involved_campaigns($store_receivers = true) {
        $args = array(
            'post_type'     => 'pcma_email_campaigns', 
            'post_status'   => array('publish', 'future'), 
            'posts_per_page'=> -1, 
            'order'         => 'ASC',
            'orderby'       => 'date',
            
            'date_query' => array(
                'before'    => current_time('c'),
                'inclusive' => true,
            ),
            'meta_query' => array(
                'relation' => 'AND',
                array(
                    'key'       => 'pcma_email_campaign_done',
                    'compare'   => 'NOT EXISTS'
                ),
                array(
                    'key'       => 'pcma_email_campaign_paused',
                    'compare'   => 'NOT EXISTS'
                ),
            )
        );
        
        
        // PCMA-FILTER - allow get_involved_campaigns() WP_query filter
        $args = apply_filters('pcma_get_involved_campaigns_query', $args);
            
        $query = new WP_Query($args);
        
        $to_return = array();
        if(is_array($query->posts)) {
            foreach($query->posts as $cmp) {
                
                $to_return[ $cmp->ID ] = array(
                    'status'    => $cmp->post_status,
                    'content'   => $cmp->post_content,
                );
                
                
                // to_return array key > meta-key
                $assoc = array(
                    'title'     => 'pcma_ecb_title',
                    'template'  => 'pcma_ecb_template',
                    'attach'    => 'pcma_ecb_attach',
                    
                    'receivers'         => 'pcma_ecb_receivers', // array of user IDs
                    'extra_user_ids'    => 'pcma_ecb_extra_user_ids', // array of extra user IDs
                    'extra_emails'      => 'pcma_ecb_extra_emails', // array of extra e-mails
                    
                    'start_date'=> 'pcma_ecb_start_time', // datetime of when campaign started
                    'processed' => 'pcma_ecb_receivers_processed', // array (user_id (or extra_email) => timestamp) of processed receivers
                    'failed'    => 'pcma_ecb_receivers_failed', // array (user_id (or extra_email) => reason) of failed submissions
                );
                
                $to_return[$cmp->ID] = array_merge(
                    $to_return[$cmp->ID], 
                    self::get_campaign_metas($cmp->ID, $assoc)
                );
                
                
                // if there are no receivers, fetch them and store
                if(!is_array($to_return[ $cmp->ID ]['receivers'])) {
                    $rec_ids_list = array();
                    
                    $receivers = self::get_involved_users($cmp->ID);
                    foreach($receivers as $rec) {
                        $rec_ids_list[] = $rec['id'];    
                    }
                        
                    // concat specific user IDs
                    if(!empty($to_return[ $cmp->ID ]['extra_user_ids'])) {
                        $rec_ids_list = array_merge($rec_ids_list, $to_return[ $cmp->ID ]['extra_user_ids']);
                        $rec_ids_list = array_values(array_unique($rec_ids_list));

                        unset($to_return[ $cmp->ID ]['extra_user_ids']);
                    }
                    asort($rec_ids_list, SORT_NATURAL);
                        
                        
                    $to_return[ $cmp->ID ]['receivers'] = $rec_ids_list;
                    if($store_receivers)  {
                        update_post_meta($cmp->ID, 'pcma_ecb_receivers', $rec_ids_list);
                    }
                }
            }
        }
        
        return $to_return;
    }
    
    

    
    /* 
     * Given campaign attachment IDs, returns an array made for pcma_send_mail() 
     */
    protected function get_attach_data($attach_ids) {
        if(empty($attach_ids)) {
            return array();    
        }
        
        $upl_dir = wp_upload_dir();
        $upl_dir = $upl_dir['basedir'];
        
        $attach_data = array();
        foreach($attach_ids as $aid) {
            
            if(isset($this->attach_data[$aid])) {
                $file = $this->attach_data[$aid]; 
                $attach_data[ $file['path'] ] = $file['name'];
                continue;
            }
            
            $file_path = get_attached_file($aid);
            if($file_path) {
                $path_arr = explode('/', $file_path);
                $filename = end($path_arr);
                
                $this->attach_data[$aid] = array(
                    'path' => $file_path,
                    'name' => $filename
                );
                $attach_data[$file_path] = $filename;
            } 
        }
        
        return $attach_data;
    }
    
    
    
    
    
    /* process campaigns */
    public function process_campaigns() {
        global $wpdb, $pc_users;

        foreach($this->get_involved_campaigns() as $cmp_id => $cmp) {

            // can send further e-mails?
            $how_many_can_send = $this->emails_limit_val - $this->emails_sent_in_past_period;
            if($how_many_can_send < 1) {
                break;    
            }
            
            $tot_receivers = array_merge($cmp['receivers'], $cmp['extra_emails']);
            $tot_receivers_count = count($tot_receivers);
            
            if(!$tot_receivers_count) {
                continue;    
            }
            
            // set start datetime
            if(!$cmp['start_date']) {
                update_post_meta($cmp_id, 'pcma_ecb_start_time', gmdate('Y-m-d H:i:s'));
            }
            
            // be sure post is set to "poublish"
            if($cmp['status'] != 'publish') {
                $wpdb->query(
                    $wpdb->prepare("UPDATE ". $wpdb->posts ." SET post_status = '%s' WHERE ID = %d",
                        'publish',
                        $cmp_id
                    )
                );    
            }
            
            // create session receivers array basing on how many e-mails can still be sent
            $new_receivers = array_slice($tot_receivers, count($cmp['processed']), $how_many_can_send); 
            
            // get users data
            $args = array(
                'user_id'   => $new_receivers,
                'limit'     => count($new_receivers),
                'to_get'    => array('id', 'email', 'name', 'surname', 'username', 'tel', 'categories'),
            );
            $user_data = $pc_users->get_users($args);
            
            
            // set user id as array key for quicker usage
            $reindexed_ud = array();
            foreach($user_data as $ud) {
                $reindexed_ud[ $ud['id'] ] = $ud;        
            }
            $user_data = $reindexed_ud;
            unset($reindexed_ud);

            //// send
            $failed = array();
            foreach($new_receivers as $receiver) {
                $cmp['processed'][$receiver] = gmdate('U');
                
                $email = $receiver;
                $ud = false;
                
                if(is_numeric($receiver)) {
                    if(!isset($user_data[$receiver])) {
                        $cmp['failed'][$receiver] = 'user not found';
                        continue;
                    }
                    
                    $ud = $user_data[$receiver];
                    $email = $ud['email'];
                }
                
                if(!$this->send_email($cmp, $email, $ud)) {
                    $cmp['failed'][$receiver] = $GLOBALS['pcma_wp_mail_error'];
                    continue;
                }
                
                else {
                    // PCMA-ACTION - trigger action when campaign e-mail is successfully sent to user - passes campaig ID and user ID
                    if(is_numeric($receiver)) {
                        do_action('pcma_email_campaign_sent_to_user', $cmp_id, $receiver);
                    }
                }
                
                $this->emails_sent_now++;
                
                
                //// update the database in real time to avoid troubles with server max exec time
                
                // processed - updaated campaign stats
                update_post_meta($cmp_id, 'pcma_ecb_receivers_processed', pc_static::compress_data($cmp['processed']));
                update_post_meta($cmp_id, 'pcma_ecb_receivers_failed', pc_static::compress_data($cmp['failed']));

                if(count($cmp['processed']) && count($cmp['processed']) >= $tot_receivers_count) {
                    update_post_meta($cmp_id, 'pcma_email_campaign_done', 1);

                    // PCMA-ACTION - trigger action when campaign is completed
                    do_action('pcma_email_campaign_done', $cmp_id);
                }
                
                // update emails sent in the past hour
                $this->update_esipp_transients();
            }
        }
        
        return true;
    }
    
    
    
    
    /*
     * Send the campaign e-mail to a single receiver
     *
     * @param (array) $campaign_data - campaign data passed by process_campaigns()
     * @param (string) $email - receiver's email
     * @param (array|false) receiver's user data (false if is a direct email)
     *
     * @return (bool) the sending result
     */
    protected function send_email($campaign_data, $email, $user_data = false) {
        $cmp = $campaign_data;
        $ud = $user_data;
        
        $mail_title = ($ud) ? pcma_replace_placeholders($ud['id'], $cmp['title'], $ud) : pcma_replace_basic_placeholders($cmp['title']);
        $mail_txt   = ($ud) ? pcma_replace_placeholders($ud['id'], $cmp['content'], $ud) : pcma_replace_basic_placeholders($cmp['title']);
        $attach     = $this->get_attach_data($cmp['attach']);

        $mail_txt = pcma_apply_mail_templates($cmp['template'], $mail_txt);
        $username = ($ud) ? $ud['username'] : $email;
        
        return pcma_send_mail(
            $username, 
            $email, 
            $mail_title, 
            $mail_txt, 
            $from_name = false, 
            $from_mail = false, 
            $reply_to_name = false, 
            $reply_to_mail = false, 
            $attach
        ); 
    }
    
    
    
    
    /* 
     * update pcma_ec_emails_sent_in_past_hour transient value without changing expration 
     */
    protected function update_esipp_transients() {
        $periods = array('hourly', 'daily', 'monthly');
        
        foreach($periods as $p) {
            $target_transient_name = str_replace('%PERIOD%', $p, self::$esipp_transient_name);
            $transient_val = (int)get_transient($target_transient_name);
            
            $new_val = $transient_val + $this->emails_sent_now;
            
            if(!$transient_val) {
                if($p == 'hourly')      {$transient_time = HOUR_IN_SECONDS;}
                elseif($p == 'daily')   {$transient_time = DAY_IN_SECONDS;}
                elseif($p == 'monthly') {$transient_time = MONTH_IN_SECONDS;}

                set_transient($target_transient_name, $new_val, $transient_time);
            }
            else {
                update_option('_transient_'. $target_transient_name, $new_val);    
            }
        }
    }
    
}



add_action('init', 'pcma_email_campaigns::register_cpt_n_ct', 999);
add_filter('post_updated_messages', 'pcma_email_campaigns::cpt_updated_messages');

add_filter('cron_schedules','pcma_email_campaigns::set_schedule_timing');
add_action('wp_loaded', 'pcma_email_campaigns::register_cron_job');
add_action('pcma_process_campaigns', 'pcma_email_campaigns::wp_schedule_process_campaigns');
add_action('pvtcont_init', 'pcma_email_campaigns::manual_campaigns_process');

